import { useState } from 'react';
import { Play, FileText } from 'lucide-react';
import { SimulationConfig } from '../../types';

interface SimulationDataGeneratorProps {
  config: SimulationConfig;
  onUpdate: (config: SimulationConfig) => void;
}

export function SimulationDataGenerator({ config, onUpdate }: SimulationDataGeneratorProps) {
  const [description, setDescription] = useState(config.description || '');
  const [sampleOutput, setSampleOutput] = useState(config.sampleOutput || '');

  const handleGenerate = () => {
    const mockOutput = `{
  "timestamp": "${new Date().toISOString()}",
  "emissions": {
    "scope1": 36297.2,
    "scope2": 8529.8,
    "scope3": 1616.4
  },
  "sensors": {
    "temperature": 22.5,
    "co2": 420,
    "humidity": 45
  },
  "zones": ["L1-A", "L1-B", "L2-C"]
}`;
    setSampleOutput(mockOutput);
    onUpdate({ ...config, description, sampleOutput: mockOutput });
  };

  return (
    <div className="space-y-6">
      <div>
        <label className="block text-sm font-medium text-gray-700 mb-2">
          Description
        </label>
        <textarea
          value={description}
          onChange={(e) => {
            setDescription(e.target.value);
            onUpdate({ ...config, description: e.target.value });
          }}
          rows={4}
          className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
          placeholder="Enter description for the simulation data generator..."
        />
      </div>

      <button
        onClick={handleGenerate}
        className="w-full flex items-center justify-center gap-2 px-4 py-3 bg-gradient-to-r from-primary to-accent text-white rounded-lg hover:shadow-md transition-all duration-300 font-medium"
      >
        <Play className="w-5 h-5" />
        Generate Sample Data
      </button>

      {sampleOutput && (
        <div>
          <div className="flex items-center gap-2 mb-2">
            <FileText className="w-4 h-4 text-gray-500" />
            <label className="text-sm font-medium text-gray-700">Example Output</label>
          </div>
          <pre className="w-full p-4 bg-gray-50 border border-gray-200 rounded-lg text-xs overflow-x-auto">
            {sampleOutput}
          </pre>
        </div>
      )}
    </div>
  );
}

