import React, { useEffect, useRef } from 'react';

interface Zone {
    id: string;
    label: string;
    svgPath: string;
}

interface BuildingMapProps {
    zonesData: Zone[];
    sensorsData: any[];
    sourcesData: any[];
    liveMode: boolean;
    onZoneSelect: (zoneId: string) => void;
    selectedZoneId: string | null;
}

const BuildingMap: React.FC<BuildingMapProps> = ({
    zonesData,
    sensorsData,
    sourcesData,
    liveMode,
    onZoneSelect,
    selectedZoneId,
}) => {
    const canvasRef = useRef<HTMLCanvasElement>(null);

    // Canvas Overlay Animation (Heatmap/Particles)
    useEffect(() => {
        const canvas = canvasRef.current;
        if (!canvas) return;
        const ctx = canvas.getContext('2d');
        if (!ctx) return;

        let animationFrameId: number;

        const render = () => {
            ctx.clearRect(0, 0, canvas.width, canvas.height);

            // Simulate heatmap overlay
            if (liveMode) {
                ctx.fillStyle = 'rgba(255, 0, 0, 0.1)';
                // Draw some random heatmap blobs based on sensor data (simulated)
                sensorsData.forEach((sensor, index) => {
                    const x = (index * 100) % canvas.width;
                    const y = (index * 50) % canvas.height;
                    ctx.beginPath();
                    ctx.arc(x + 50, y + 50, 40, 0, 2 * Math.PI);
                    ctx.fill();
                });
            }

            animationFrameId = requestAnimationFrame(render);
        };

        render();

        return () => cancelAnimationFrame(animationFrameId);
    }, [liveMode, sensorsData]);

    return (
        <div className="relative w-full h-full bg-gray-50 rounded-xl overflow-hidden border border-gray-200" role="application" aria-label="Building Digital Twin Map">
            {/* SVG Layer */}
            <svg
                viewBox="0 0 800 600"
                className="absolute inset-0 w-full h-full z-10"
                preserveAspectRatio="xMidYMid meet"
            >
                {zonesData.map((zone) => (
                    <g
                        key={zone.id}
                        onClick={() => onZoneSelect(zone.id)}
                        className={`cursor-pointer transition-all duration-300 hover:opacity-80 ${selectedZoneId === zone.id ? 'opacity-100 stroke-primary stroke-2' : 'opacity-60 hover:opacity-80'
                            }`}
                        aria-label={`Select Zone ${zone.label}`}
                    >
                        <path
                            d={zone.svgPath}
                            fill={selectedZoneId === zone.id ? '#10B981' : '#CBD5E1'} // Green if selected, Gray otherwise
                            stroke="white"
                            strokeWidth="2"
                        />
                        <text
                            x="50%" // Placeholder positioning, real paths would need centroids
                            y="50%"
                            className="text-xs font-bold fill-gray-700 pointer-events-none"
                            textAnchor="middle"
                        >
                            {zone.label}
                        </text>
                    </g>
                ))}
            </svg>

            {/* Canvas Overlay Layer */}
            <canvas
                ref={canvasRef}
                width={800}
                height={600}
                className="absolute inset-0 w-full h-full z-20 pointer-events-none opacity-50"
            />

            {/* Sources Markers */}
            {sourcesData.map((source) => (
                <div
                    key={source.id}
                    className="absolute z-30 w-4 h-4 bg-red-500 rounded-full animate-pulse border-2 border-white shadow-lg"
                    style={{ top: '30%', left: '40%' }} // Placeholder positions
                    title={`${source.type}: ${source.tCO2e} tCO2e`}
                ></div>
            ))}
        </div>
    );
};

export default BuildingMap;
