import { Helmet } from 'react-helmet-async';
import { motion } from 'framer-motion';
import { Phone, Globe, Target, Heart, Zap, ShieldCheck, Building2 } from 'lucide-react';
import Navbar from './Navbar';
import Footer from './Footer';

const values = [
    {
        icon: Globe,
        title: "Our Vision",
        description: "To lead the digital transformation of compliance and sustainability across Saudi Arabia by providing fully automated, real-time intelligence.",
        color: "text-primary",
        bg: "bg-primary/10",
        border: "border-primary/20"
    },
    {
        icon: Target,
        title: "Our Mission",
        description: "To empower organizations with a unified platform that reduces risk, enhances environmental performance, accelerates reporting, and simplifies compliance.",
        color: "text-accent",
        bg: "bg-accent/10",
        border: "border-accent/20"
    },
    {
        icon: Heart,
        title: "Our Values",
        description: "Transparency • Innovation • Accuracy • Responsibility • Sustainability",
        color: "text-purple-400",
        bg: "bg-purple-400/10",
        border: "border-purple-400/20"
    }
];

const features = [
    {
        icon: Zap,
        title: "AI-Powered Automation",
        description: "Intelligent data processing and predictive analytics."
    },
    {
        icon: Globe,
        title: "Integrated Sustainability",
        description: "Comprehensive tools for carbon, ESG, and environmental management."
    },
    {
        icon: ShieldCheck,
        title: "Unified Compliance",
        description: "Centralized infrastructure for all regulatory requirements."
    }
];

export default function AboutPage() {
    return (
        <div className="bg-dark min-h-screen text-white overflow-x-hidden">
            <Helmet>
                <title>About Us - Mutqin Pro</title>
                <meta name="description" content="Learn about Mutqin Pro's mission to unify compliance and sustainability in Saudi Arabia through AI-powered digital transformation." />
            </Helmet>
            <Navbar />

            {/* Hero Section */}
            <section className="relative pt-32 pb-20 overflow-hidden">
                <div className="absolute inset-0 bg-dark z-0">
                    <div className="absolute top-0 right-0 w-[600px] h-[600px] bg-primary/10 rounded-full blur-[128px]" />
                    <div className="absolute bottom-0 left-0 w-[600px] h-[600px] bg-secondary/20 rounded-full blur-[128px]" />
                </div>

                <div className="container mx-auto px-6 relative z-10 grid lg:grid-cols-2 gap-12 items-center">
                    <motion.div
                        initial={{ opacity: 0, x: -30 }}
                        animate={{ opacity: 1, x: 0 }}
                        transition={{ duration: 0.8 }}
                    >
                        <h1 className="text-4xl lg:text-6xl font-bold leading-tight mb-6">
                            About <span className="text-gradient">MUTQIN Pro</span>
                        </h1>
                        <p className="text-xl text-gray-400 mb-8 leading-relaxed">
                            MUTQIN Pro is a next-generation Saudi SaaS platform designed to unify compliance, sustainability, carbon management, quality, safety, and environmental reporting—all in one intelligent ecosystem.
                        </p>
                    </motion.div>

                    <motion.div
                        initial={{ opacity: 0, scale: 0.8, rotateY: -30 }}
                        animate={{ opacity: 1, scale: 1, rotateY: 0 }}
                        transition={{ duration: 1 }}
                        className="relative h-[400px] flex items-center justify-center perspective-1000"
                    >
                        {/* Abstract 3D Shape Representation */}
                        <div className="relative w-64 h-64">
                            <div className="absolute inset-0 bg-gradient-to-br from-primary to-secondary rounded-3xl opacity-20 animate-pulse" />
                            <div className="absolute inset-4 bg-gradient-to-tr from-accent to-primary rounded-3xl opacity-40 rotate-12" />
                            <div className="absolute inset-8 bg-gradient-to-bl from-secondary to-accent rounded-3xl opacity-60 -rotate-12" />
                            <div className="absolute inset-0 flex items-center justify-center">
                                <Building2 className="w-24 h-24 text-white drop-shadow-glow" />
                            </div>
                        </div>
                    </motion.div>
                </div>
            </section>

            {/* Vision / Mission / Values */}
            <section className="py-20 relative">
                <div className="container mx-auto px-6">
                    <div className="grid md:grid-cols-3 gap-8">
                        {values.map((item, index) => (
                            <motion.div
                                key={index}
                                initial={{ opacity: 0, y: 30 }}
                                whileInView={{ opacity: 1, y: 0 }}
                                viewport={{ once: true }}
                                transition={{ delay: index * 0.1 }}
                                whileHover={{ y: -10 }}
                                className={`group relative bg-white/5 backdrop-blur-lg border border-white/10 rounded-3xl p-8 hover:border-primary/50 transition-all duration-300 hover:shadow-glow`}
                            >
                                <div className={`w-14 h-14 rounded-2xl ${item.bg} ${item.border} border flex items-center justify-center mb-6 group-hover:scale-110 transition-transform`}>
                                    <item.icon className={`w-7 h-7 ${item.color}`} />
                                </div>
                                <h3 className="text-2xl font-bold mb-4 group-hover:text-primary transition-colors">{item.title}</h3>
                                <p className="text-gray-400 leading-relaxed">{item.description}</p>
                            </motion.div>
                        ))}
                    </div>
                </div>
            </section>

            {/* Strategic Partnerships */}
            <section className="py-20 relative">
                <div className="container mx-auto px-6 text-center">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        viewport={{ once: true }}
                        className="mb-16"
                    >
                        <h2 className="text-3xl lg:text-4xl font-bold mb-4">Strategic Partnerships</h2>
                        <p className="text-gray-400 max-w-2xl mx-auto">
                            We collaborate with leading entities, certification bodies, and sustainability organizations.
                        </p>
                    </motion.div>

                    <div className="flex justify-center items-center gap-64 py-6">
    {/* Left Logo */}
    <img
        src="/partner1.png"
        alt="Left Logo"
        className="max-h-40 w-auto object-contain"
    />

    {/* Center Logo */}
    <img
        src="/partner2.png"
        alt="Center Logo"
        className="max-h-40 w-auto object-contain"
    />

    {/* Right Logo */}
    <img
        src="/partner3.png"
        alt="Right Logo"
        className="max-h-40 w-auto object-contain"
    />
</div>

                </div>
            </section>

            {/* Why Mutqin Pro */}
            <section className="py-20 bg-gradient-to-b from-dark to-primary/5">
                <div className="container mx-auto px-6">
                    <div className="grid md:grid-cols-3 gap-8">
                        {features.map((feature, index) => (
                            <motion.div
                                key={index}
                                initial={{ opacity: 0, y: 20 }}
                                whileInView={{ opacity: 1, y: 0 }}
                                viewport={{ once: true }}
                                transition={{ delay: index * 0.1 }}
                                className="text-center p-6"
                            >
                                <div className="w-16 h-16 mx-auto bg-primary/10 rounded-full flex items-center justify-center mb-4 text-primary">
                                    <feature.icon className="w-8 h-8" />
                                </div>
                                <h3 className="text-xl font-bold mb-2">{feature.title}</h3>
                                <p className="text-gray-400 text-sm">{feature.description}</p>
                            </motion.div>
                        ))}
                    </div>
                </div>
            </section>

            {/* CTA Section */}
            <section className="py-24 relative overflow-hidden">
                <div className="absolute inset-0 bg-gradient-to-r from-secondary/50 to-primary/20" />
                <div className="container mx-auto px-6 text-center relative z-10">
                    <h2 className="text-4xl font-bold mb-4">Got Questions?</h2>
                    <p className="text-xl text-gray-300 mb-10">Schedule your consultation with MUTQIN Pro today.</p>

                    <motion.button
                        whileHover={{ y: -5 }}
                        className="group relative inline-flex items-center gap-3 px-8 py-4 rounded-xl border-2 border-primary text-primary font-bold text-lg transition-all hover:shadow-[0_0_20px_rgba(29,167,161,0.3)] bg-transparent"
                    >
                        <span className="flex items-center justify-center w-8 h-8 rounded bg-primary text-white shadow-sm group-hover:shadow-glow transition-shadow">
                            <Phone className="w-4 h-4" />
                        </span>
                        Contact Us
                    </motion.button>
                </div>
            </section>

            <Footer />
        </div>
    );
}
